//
// ViewController.m
// Cars DataGrid Sample
//
// This sample introduces how to create UIDataGrid object and  add corresponding DataGrid
// Columns that contain Picker, DatePicker,  TextField input box, Label Column,  TextView
// editor, Button Column, and other column types into a UIDataGrid on your iOS app layout.
//
// The main idea of the samples is to provide you with  a suggestion how to update values
// in other DataGrid  cells at  a row when  the current  column  cell has  been updated.
// For this case, when we need to modify corresponding cells we are implementing specific
// DataGrid Columns' value changing events.
//
// Also the code example gives hint how to deal with source data when you need manipulate
// and update a set of cells/rows in the UIDataGrid interface  and store the updates back
// into the datasource mutable array object. It explains how to enforce that by using the
// updateDataSource property of UIDataGrid object.
//
// Please read on-line documentation: http://rustemsoft.com/TouchControlsDoc/index.html
//
// Copyright (c) 2015 Rustemsoft LLC. All rights reserved.
//
// ======================================================================================
// Xcode 6.1.1 still has problems related to framework build and referencing.
// We have listed resolutions for couple of possible issues you may encounter:
//
// 1. When you run the Sample project for very first time the build may fail. To fix that
//   remove the reference to TouchControlsTrial.framework and re-reference to the library
//   as following:
// - In the Project Navigator, select the Sample project.
// - Select the target.
// - Select the "General" tab.
// - Open "Embedded Binaries" expander.
// - Click the + button.
// - Select the TouchControlsTrial.framework.
// - Window -> Organizer -> Delete Derived Data and then: Product -> Clean.
//
// 2. Framework Resources, especially images, can be not visible on your app layout.
//   To fix that make a reference to TouchControls_Resources folder as following:
// - In the Project Navigator, select your project main folder.
// - Right click and select "Add Files to..."
// - Select the TouchControls_Resources folder and add it.
//

#import "ViewController.h"
#define COEFF_PAD ((UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPad) ? 1.6 : 1.0)

@interface ViewController ()
{
    UIDataGrid *dgrid;
    UIButton *btnPickerColumnSample;
    UIButton *btnDatePickerColumnSample;
    UIButton *btnTextViewColumnSample;
    UIButton *btnChange;
    UIView *bottomView;
    UILabel *lblSampleTitle;
}
@end

@implementation ViewController

- (void)viewDidLoad
{
    [super viewDidLoad];
    
    // Create UIDataGrid object
    dgrid = [[UIDataGrid alloc] initWithFrame:CGRectMake(COEFF_PAD * 10,  COEFF_PAD * 40, self.view.frame.size.width - COEFF_PAD * 20, self.view.frame.size.height - COEFF_PAD * 80)];
    dgrid.delegate = self; // set UIDataGrid's delegate to self (otherwise events will not work)
    
    // Define UIDataGrid's DataSource
    // 1. NSMutableArray DataSource
    dgrid.DataSource = [self generateNSMutableArrayDataSource];
    
    // 2. Plist file DataSource
    //dgrid.DataSource = [NSMutableArray arrayWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"Cars" ofType:@"plist"]];
    
    // 3. JSON file DataSource
    //dgrid.DataSource = [self generateJSONfileDataSource];
    
    // 4. XML file DataSource
    //dgrid.DataSource = [self generateXMLfileDataSource];
    
    // Add UIDataGrid object into app's main view layout
    [self.view addSubview:dgrid];
    
    // Add bottomView panel where buttons will be hosted
    bottomView = [[UIView alloc]initWithFrame:CGRectMake(0, self.view.frame.size.height - COEFF_PAD * 30, self.view.frame.size.width, COEFF_PAD * 30)];
    [bottomView setBackgroundColor:[UIColor colorWithRed:0.2 green:0.4 blue:0.6 alpha:0.8]];
    [self.view addSubview:bottomView];
    
    // Add btnPickerColumnSample button
    btnPickerColumnSample = [[UIButton alloc] initWithFrame:CGRectMake(COEFF_PAD * 5, COEFF_PAD * 8, COEFF_PAD * 66, COEFF_PAD * 14)];
    [btnPickerColumnSample setTitleColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.19 alpha:1] forState:UIControlStateNormal];
    [btnPickerColumnSample setTitleColor:[UIColor redColor] forState:UIControlStateSelected];
    [btnPickerColumnSample setBackgroundColor:[UIColor colorWithRed:1 green:1 blue:1 alpha:0.5]];
    [btnPickerColumnSample setTitle:@"Picker Column Sample" forState:UIControlStateNormal];
    [btnPickerColumnSample.titleLabel setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6]];
    btnPickerColumnSample.contentHorizontalAlignment = UIControlContentHorizontalAlignmentCenter;
    btnPickerColumnSample.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    [btnPickerColumnSample addTarget:self action:@selector(btnPickerColumnSampleClicked:) forControlEvents:UIControlEventTouchUpInside];
    btnPickerColumnSample.selected = YES;
    [bottomView addSubview:btnPickerColumnSample];
    
    // Add btnDatePickerColumnSample button
    btnDatePickerColumnSample = [[UIButton alloc] initWithFrame:CGRectMake(COEFF_PAD * 75, COEFF_PAD * 8, COEFF_PAD * 75, COEFF_PAD * 14)];
    [btnDatePickerColumnSample setTitleColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.19 alpha:1] forState:UIControlStateNormal];
    [btnDatePickerColumnSample setTitleColor:[UIColor redColor] forState:UIControlStateSelected];
    [btnDatePickerColumnSample setBackgroundColor:[UIColor colorWithRed:1 green:1 blue:1 alpha:0.5]];
    [btnDatePickerColumnSample setTitle:@"DatePicker Column Sample" forState:UIControlStateNormal];
    [btnDatePickerColumnSample.titleLabel setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6]];
    btnDatePickerColumnSample.contentHorizontalAlignment = UIControlContentHorizontalAlignmentCenter;
    btnDatePickerColumnSample.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    [btnDatePickerColumnSample addTarget:self action:@selector(btnDatePickerColumnSampleClicked:) forControlEvents:UIControlEventTouchUpInside];
    btnDatePickerColumnSample.selected = NO;
    [bottomView addSubview:btnDatePickerColumnSample];
    
    // Add btnTextViewColumnSample button
    btnTextViewColumnSample = [[UIButton alloc] initWithFrame:CGRectMake(COEFF_PAD * 160, COEFF_PAD * 8, COEFF_PAD * 75, COEFF_PAD * 14)];
    [btnTextViewColumnSample setTitleColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.19 alpha:1] forState:UIControlStateNormal];
    [btnTextViewColumnSample setTitleColor:[UIColor redColor] forState:UIControlStateSelected];
    [btnTextViewColumnSample setBackgroundColor:[UIColor colorWithRed:1 green:1 blue:1 alpha:0.5]];
    [btnTextViewColumnSample setTitle:@"TextView Column Sample" forState:UIControlStateNormal];
    [btnTextViewColumnSample.titleLabel setFont:[UIFont fontWithName:@"Georgia" size:COEFF_PAD * 6]];
    btnTextViewColumnSample.contentHorizontalAlignment = UIControlContentHorizontalAlignmentCenter;
    btnTextViewColumnSample.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    [btnTextViewColumnSample addTarget:self action:@selector(btnTextViewColumnSampleClicked:) forControlEvents:UIControlEventTouchUpInside];
    btnTextViewColumnSample.selected = NO;
    [bottomView addSubview:btnTextViewColumnSample];
    
    // Add btnChange button
    btnChange = [[UIButton alloc] initWithFrame:CGRectMake(self.view.frame.size.width - COEFF_PAD * 75, COEFF_PAD * 7, COEFF_PAD * 65, COEFF_PAD * 16)];
    [btnChange setTitleColor:[UIColor colorWithRed:0.95 green:0.1 blue:0.19 alpha:1] forState:UIControlStateNormal];
    [btnChange setBackgroundColor:[UIColor colorWithRed:1 green:1 blue:1 alpha:0.5]];
    [btnChange setImage:[UIImage imageNamed:@"change.gif"] forState:UIControlStateNormal];
    [btnChange setTitle:@"Analyze" forState:UIControlStateNormal];
    [btnChange.titleLabel setFont:[UIFont fontWithName:@"HelveticaNeue-CondensedBold" size:COEFF_PAD * 10]];
    btnChange.contentHorizontalAlignment = UIControlContentHorizontalAlignmentLeft;
    btnChange.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    btnChange.layer.cornerRadius = 7;
    btnChange.layer.borderColor = [UIColor whiteColor].CGColor;
    btnChange.layer.borderWidth = COEFF_PAD * 0.2;
    [btnChange addTarget:self action:@selector(btnChangeClicked:) forControlEvents:UIControlEventTouchUpInside];
    [bottomView addSubview:btnChange];
    
    // Define Sample Title label
    lblSampleTitle = [[UILabel alloc] initWithFrame:CGRectMake(COEFF_PAD * 10, COEFF_PAD * 30, COEFF_PAD * 120, COEFF_PAD * 10)];
    [lblSampleTitle setTextColor:[UIColor purpleColor]];
    [lblSampleTitle setFont:[UIFont fontWithName:@"HelveticaNeue-CondensedBold" size:COEFF_PAD * 9]];
    [self.view addSubview:lblSampleTitle];
    
    // Set up UIDataGrid object's layout to show Picker Column Sample
    [self btnPickerColumnSampleClicked:btnPickerColumnSample];
}

// rotate iPhone/iPad
- (void)willRotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration {
    // Define main screen width and height
    CGFloat mainWidth = ((toInterfaceOrientation == UIInterfaceOrientationLandscapeLeft || toInterfaceOrientation == UIInterfaceOrientationLandscapeRight) ? self.view.frame.size.height : self.view.frame.size.width);
    CGFloat mainHeight = ((toInterfaceOrientation == UIInterfaceOrientationLandscapeLeft || toInterfaceOrientation == UIInterfaceOrientationLandscapeRight) ? self.view.frame.size.width : self.view.frame.size.height);
    // Set btnChange x coordinate depending on interface orientation
    CGFloat originInX = mainWidth - btnChange.frame.size.width - COEFF_PAD * 10;
    // Set the btnChange's x offset
    bottomView.frame = CGRectMake(0, mainHeight - COEFF_PAD * 30, mainWidth, COEFF_PAD * 30);
    btnChange.frame = CGRectMake(originInX, COEFF_PAD * 7, btnChange.frame.size.width, btnChange.frame.size.height);
}

// Click btnPickerColumnSample button
- (void)btnPickerColumnSampleClicked:(UIButton*)btn {
    // Reset buttons' 'Selected' status
    btnPickerColumnSample.selected = NO;
    btnDatePickerColumnSample.selected = NO;
    btnTextViewColumnSample.selected = NO;
    btn.selected = YES;
    lblSampleTitle.text = btn.titleLabel.text;
    
    // Set up UIDataGrid object's layout to show Picker Column Sample
    // Adding Columns
    NSMutableArray *cols = [[NSMutableArray alloc] init];
    
    // Define DataGridImageColumn object
    DataGridImageColumn *colImg = [[DataGridImageColumn alloc] init];
    colImg.dataFieldIndex = 8;
    colImg.Title = @"Logo";
    colImg.Width = COEFF_PAD * 50;
    [colImg.header setTextAlignment:NSTextAlignmentCenter];
    colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40);
    colImg.cellLayoutPosition = LayoutPositionRightCenter;
    colImg.cellLayoutPositionX = COEFF_PAD * 5;
    colImg.cellLayoutPositionY = COEFF_PAD * 10;
    [cols addObject:colImg];
    
    // Define DataGridLabelColumn object
    DataGridLabelColumn *col = [[DataGridLabelColumn alloc] init];
    col.dataFieldIndex = 1;
    col.Width = COEFF_PAD * 60;
    col.Title = @"Corporation";
    [col setBackgroundColor:[UIColor colorWithRed:0.01 green:0.01 blue:0.5 alpha:0.3]];
    [col.header setTextAlignment:NSTextAlignmentCenter];
    col.HeaderBackgroundColor = [UIColor colorWithRed:0.01 green:0.01 blue:0.5 alpha:0.3];
    col.HeaderForegroundColor = [UIColor blackColor];
    col.HeaderFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 7];
    col.borderColor = [UIColor darkTextColor];
    col.borderWidth = COEFF_PAD * .3;
    // Define a custom cell size and cell's position
    col.cellSize = CGSizeMake(COEFF_PAD * 52, COEFF_PAD * 16);
    col.cellLayoutPosition = LayoutPositionCenter;
    [col.cellTemplate setTextAlignment:NSTextAlignmentCenter];
    [col.cellTemplate setTextColor:[UIColor darkTextColor]];
    [col.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8]];
    [cols addObject:col];
    
    // Define DataGridPickerColumn object
    DataGridPickerColumn *colpkr = [[DataGridPickerColumn alloc] init];
    colpkr.dataFieldIndex = 1;
    // Define Data source for picker items
    NSMutableArray *dsr = [[NSMutableArray alloc] init];
    [dsr addObject:@""];
    [dsr addObject:@"Ford"];
    [dsr addObject:@"Mitsubishi"];
    [dsr addObject:@"Toyota"];
    [dsr addObject:@"Honda"];
    [dsr addObject:@"Subaru"];
    [dsr addObject:@"Dodge"];
    [dsr addObject:@"Jeep"];
    [dsr addObject:@"Volkswagen"];
    [dsr addObject:@"Nissan"];
    [dsr addObject:@"Chevrolet"];
    [dsr addObject:@"Fiat"];
    [dsr addObject:@"Hyundai"];
    [dsr addObject:@"Kia"];
    [dsr addObject:@"Mercedes"];
    [dsr addObject:@"Peugeot"];
    [dsr addObject:@"Renault"];
    colpkr.pickerDataSource = dsr;
    colpkr.Title = @"Select";
    colpkr.hidden = NO;
    colpkr.pickerFrameColor = greenPickerFrameColor;
    [colpkr.header setTextAlignment:NSTextAlignmentCenter];
    colpkr.Width = COEFF_PAD * 120;
    // Define a custom cell size
    colpkr.cellSize = CGSizeMake(COEFF_PAD * 100, COEFF_PAD * 60);
    // Define a custom UILabel object as cellLabelTemplate
    UILabel *lblRow = [[UILabel alloc] init];
    // Assign Text Alignment
    [lblRow setTextAlignment:NSTextAlignmentCenter];
    [lblRow setTextColor:[UIColor redColor]];
    [lblRow setFont:[UIFont fontWithName:@"HelveticaNeue-CondensedBold" size:COEFF_PAD * 17]];
    colpkr.cellLabelTemplate = lblRow;
    colpkr.pickerRowHeight = COEFF_PAD * 25;
    [cols addObject:colpkr];
    
    // Define DataGridSwitchColumn object
    DataGridSwitchColumn *colSwh = [[DataGridSwitchColumn alloc] init];
    colSwh.dataFieldIndex = 12;
    colSwh.Title = @"Access";
    colSwh.header.textAlignment = NSTextAlignmentLeft;
    colSwh.hidden = NO;
    [colSwh setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.1 alpha:0.1]];
    colSwh.cellSize = CGSizeMake(COEFF_PAD * 50, COEFF_PAD * 30);
    colSwh.onTintColor = [UIColor greenColor];
    colSwh.borderWidth = 0;
    [cols addObject:colSwh];
    
    // Define DataGridTextFieldColumn object
    DataGridTextFieldColumn *coltxt = [[DataGridTextFieldColumn alloc] init];
    coltxt.dataFieldIndex = 6;
    coltxt.Width = 115;
    coltxt.Title = @"Phone";
    [coltxt.header setTextAlignment:NSTextAlignmentCenter];
    coltxt.autocapitalizationType = UITextAutocapitalizationTypeNone;
    coltxt.borderStyle = UITextBorderStyleRoundedRect;
    coltxt.clearButtonMode = UITextFieldViewModeNever;
    coltxt.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    coltxt.contentHorizontalAlignment = UIControlContentHorizontalAlignmentFill;
    coltxt.keyboardType = UIKeyboardTypeAlphabet;
    coltxt.placeholder = @"";
    coltxt.returnKeyType = UIReturnKeyDone;
    coltxt.TextAlignment = NSTextAlignmentNatural;
    coltxt.textColor = [UIColor blackColor];
    [coltxt.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 12]];
    coltxt.recognizeCellTapped = NO;
    [cols addObject:coltxt];
    
    // Define DataGridButtonColumn object
    DataGridButtonColumn *colBtn = [[DataGridButtonColumn alloc] init];
    colBtn.dataFieldIndex = 10;
    colBtn.AutoSizeType = None;
    colBtn.Title = @"ZipCode";
    colBtn.font = [UIFont fontWithName:@"Georgia-Italic" size:COEFF_PAD * 10];
    [colBtn.header setTextAlignment:NSTextAlignmentCenter];
    colBtn.cellSize = CGSizeMake(70, 40);
    colBtn.cellLayoutPosition = LayoutPositionRightCenter;
    colBtn.cellTemplate.layer.cornerRadius = 7;
    colBtn.cellTemplate.layer.borderColor = [UIColor grayColor].CGColor;
    colBtn.cellTemplate.layer.borderWidth = COEFF_PAD * 3.2;
    colBtn.cellTemplate.layer.backgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2].CGColor;
    [cols addObject:colBtn];
    
    // Add Columns to Column collection
    dgrid.Columns = cols;
    
    // Assign UIDataGrid main properties
    dgrid.ColumnsWidth = COEFF_PAD * 110;
    // Datagrid's main colors
    dgrid.backgroundColor = [UIColor clearColor];
    dgrid.foregroundColor = [UIColor purpleColor];
    // Color of every other row
    dgrid.alternatingBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.1];
    // Specify properties of headers
    dgrid.HeaderBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2];
    dgrid.HeaderForegroundColor = [UIColor whiteColor];
    dgrid.HeaderFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8];
    dgrid.HeaderBorderColor = [UIColor lightGrayColor];
    dgrid.HeaderBorderWidth = COEFF_PAD * .3;
    dgrid.HeaderScrollHide = NO;
    
    dgrid.rowBorderHeight = 1;
    dgrid.rowBorderColor = [UIColor cyanColor];
    dgrid.RowHeight = COEFF_PAD * 70;
    dgrid.datagridFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 4];
    dgrid.updateDataSource = YES;
    
    // Bind to populate data from DataSource into UIDataGrid object
    [dgrid DataBind];
}

// Click btnDatePickerColumnSample button
- (void)btnDatePickerColumnSampleClicked:(UIButton*)btn {
    // Reset buttons' 'Selected' status
    btnPickerColumnSample.selected = NO;
    btnDatePickerColumnSample.selected = NO;
    btnTextViewColumnSample.selected = NO;
    btn.selected = YES;
    lblSampleTitle.text = btn.titleLabel.text;
    
    // Set up UIDataGrid object's layout to show DatePicker Column Sample
    // Adding Columns
    NSMutableArray *cols = [[NSMutableArray alloc] init];
    
    // Define DataGridImageColumn object
    DataGridImageColumn *colImg = [[DataGridImageColumn alloc] init];
    colImg.dataFieldIndex = 8;
    colImg.Title = @"Logo";
    colImg.Width = COEFF_PAD * 50;
    [colImg.header setTextAlignment:NSTextAlignmentCenter];
    colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40);
    colImg.cellLayoutPosition = LayoutPositionRightCenter;
    colImg.cellLayoutPositionX = COEFF_PAD * 5;
    colImg.cellLayoutPositionY = COEFF_PAD * 10;
    [cols addObject:colImg];
    
    // Define DataGridLabelColumn object
    DataGridLabelColumn *col = [[DataGridLabelColumn alloc] init];
    col.dataFieldIndex = 11;
    col.Width = COEFF_PAD * 30;
    col.Title = @"%%";
    [col.header setTextAlignment:NSTextAlignmentCenter];
    [col.cellTemplate setTextAlignment:NSTextAlignmentCenter];
    [col.cellTemplate setTextColor:[UIColor darkTextColor]];
    [col.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8]];
    [cols addObject:col];
    
    // Define DataGridSliderColumn object
    DataGridSliderColumn *colSldr = [[DataGridSliderColumn alloc] init];
    colSldr.dataFieldIndex = 11;
    colSldr.Title = @"Availability";
    colSldr.header.textAlignment = NSTextAlignmentCenter;
    colSldr.hidden = NO;
    colSldr.Width = COEFF_PAD * 90;
    colSldr.cellSize = CGSizeMake(COEFF_PAD * 80, COEFF_PAD * 50);
    [colSldr setBackgroundColor:[UIColor colorWithRed:0.1 green:0.1 blue:0.1 alpha:0.1]];
    colSldr.borderWidth = 0;
    colSldr.minimumValue = 0;
    colSldr.maximumValue = 110;
    colSldr.cellTemplate.tintColor = [UIColor redColor];
    [cols addObject:colSldr];
    
    // Define DataGridDatePickerColumn object
    DataGridDatePickerColumn *colDtPkr = [[DataGridDatePickerColumn alloc] init];
    colDtPkr.dataFieldIndex = 5;
    colDtPkr.Width = COEFF_PAD * 280;
    colDtPkr.Title = @"Date";
    colSldr.header.textAlignment = NSTextAlignmentCenter;
    colDtPkr.hidden = NO;
    colDtPkr.pickerFrameColor = bluePickerFrameColor;
    colDtPkr.cellTemplate.datePickerMode = UIDatePickerModeDate;
    colDtPkr.dateFormat = @"MM/dd/yyyy";
    [cols addObject:colDtPkr];
    
    // Define DataGridButtonColumn object
    DataGridButtonColumn *colBtn = [[DataGridButtonColumn alloc] init];
    colBtn.dataFieldIndex = 10;
    colBtn.AutoSizeType = None;
    colBtn.Title = @"Press to Start!";
    colBtn.font = [UIFont fontWithName:@"Georgia" size:COEFF_PAD * 10];
    [colBtn.header setTextAlignment:NSTextAlignmentCenter];
    colBtn.cellSize = CGSizeMake(COEFF_PAD * 70, COEFF_PAD * 40);
    colBtn.cellTemplate.layer.cornerRadius = 7;
    colBtn.cellTemplate.layer.borderColor = [UIColor colorWithRed:0.2 green:0.3 blue:0.2 alpha:0.2].CGColor;
    colBtn.cellTemplate.layer.borderWidth = COEFF_PAD * 3.2;
    colBtn.cellTemplate.layer.backgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2].CGColor;
    [cols addObject:colBtn];
    
    // Add Columns to Column collection
    dgrid.Columns = cols;
    
    // Assign UIDataGrid main properties
    dgrid.ColumnsWidth = COEFF_PAD * 110;
    // Datagrid's main colors
    [dgrid setBackgroundColor:[UIColor colorWithRed:0.9 green:0.9 blue:0.9 alpha:0.3]];
    dgrid.foregroundColor = [UIColor purpleColor];
    // Color of every other row
    dgrid.alternatingBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2];
    // Specify properties of headers
    dgrid.HeaderBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2];
    dgrid.HeaderForegroundColor = [UIColor whiteColor];
    dgrid.HeaderFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8];
    dgrid.HeaderBorderColor = [UIColor lightGrayColor];
    dgrid.HeaderBorderWidth = COEFF_PAD * .3;
    dgrid.HeaderScrollHide = NO;
    
    dgrid.cellBorderColor = [UIColor cyanColor];
    dgrid.cellBorderWidth = .5;
    dgrid.rowBorderColor = [UIColor clearColor];
    dgrid.rowBorderHeight = COEFF_PAD * 1.5;
    dgrid.RowHeight = COEFF_PAD * 70;
    dgrid.datagridFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 4];
    dgrid.updateDataSource = YES;
    
    // Bind to populate data from DataSource into UIDataGrid object
    [dgrid DataBind];
}

// Click btnTextViewColumnSample button
- (void)btnTextViewColumnSampleClicked:(UIButton*)btn {
    // Reset buttons' 'Selected' status
    btnPickerColumnSample.selected = NO;
    btnDatePickerColumnSample.selected = NO;
    btnTextViewColumnSample.selected = NO;
    btn.selected = YES;
    lblSampleTitle.text = btn.titleLabel.text;
    
    // Set up UIDataGrid object's layout to show TextView Column Sample
    // Adding Columns
    NSMutableArray *cols = [[NSMutableArray alloc] init];
    
    // Define DataGridImageColumn object
    DataGridImageColumn *colImg = [[DataGridImageColumn alloc] init];
    colImg.dataFieldIndex = 8;
    colImg.Title = @"Logo";
    colImg.Width = COEFF_PAD * 50;
    [colImg.header setTextAlignment:NSTextAlignmentCenter];
    colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40);
    colImg.cellLayoutPosition = LayoutPositionRightCenter;
    colImg.cellLayoutPositionX = COEFF_PAD * 5;
    colImg.cellLayoutPositionY = COEFF_PAD * 10;
    [cols addObject:colImg];
    
    // Define DataGridLabelColumn object
    DataGridLabelColumn *col = [[DataGridLabelColumn alloc] init];
    col.dataFieldIndex = 1;
    col.AutoSizeType = CellContentSize;
    col.Title = @"Corporation";
    [col.header setTextAlignment:NSTextAlignmentCenter];
    [col.cellTemplate setTextAlignment:NSTextAlignmentCenter];
    [col.cellTemplate setTextColor:[UIColor darkTextColor]];
    [col.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8]];
    [cols addObject:col];
    
    // Define DataGridTextViewColumn object
    DataGridTextViewColumn *cTxtVw = [[DataGridTextViewColumn alloc] init];
    cTxtVw.dataFieldIndex = 9;
    cTxtVw.Title = @"Remark";
    [cTxtVw.header setTextAlignment:NSTextAlignmentCenter];
    cTxtVw.autocapitalizationType = UITextAutocapitalizationTypeNone;
    cTxtVw.keyboardType = UIKeyboardTypeAlphabet;
    cTxtVw.returnKeyType = UIReturnKeyDone;
    cTxtVw.TextAlignment = NSTextAlignmentNatural;
    cTxtVw.textColor = [UIColor blackColor];
    [cTxtVw.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8]];
    cTxtVw.recognizeCellTapped = NO;
    cTxtVw.Width = COEFF_PAD * 250;
    cTxtVw.cellSize = CGSizeMake(COEFF_PAD * 230, COEFF_PAD * 70);
    cTxtVw.cellLayoutPositionX = COEFF_PAD * 5;
    cTxtVw.cellLayoutPositionY = COEFF_PAD * 10;
    [cols addObject:cTxtVw];
    
    // Define DataGridTextFieldColumn object
    DataGridTextFieldColumn *coltxt = [[DataGridTextFieldColumn alloc] init];
    coltxt.dataFieldIndex = 12;
    coltxt.Title = @"Number";
    // Format numeric input
    coltxt.numberFormatterStyle = NSNumberFormatterDecimalStyle;
    coltxt.numberWithType = numberWithFloat;
    // Header's title position
    [coltxt.header setTextAlignment:NSTextAlignmentCenter];
    coltxt.autocapitalizationType = UITextAutocapitalizationTypeNone;
    coltxt.borderStyle = UITextBorderStyleRoundedRect;
    coltxt.clearButtonMode = UITextFieldViewModeNever;
    coltxt.contentVerticalAlignment = UIControlContentVerticalAlignmentCenter;
    coltxt.contentHorizontalAlignment = UIControlContentHorizontalAlignmentFill;
    coltxt.keyboardType = UIKeyboardTypeAlphabet;
    coltxt.placeholder = @"";
    coltxt.returnKeyType = UIReturnKeyDone;
    coltxt.TextAlignment = NSTextAlignmentNatural;
    coltxt.textColor = [UIColor blackColor];
    [coltxt.cellTemplate setFont:[UIFont fontWithName:@"Verdana" size:COEFF_PAD * 12]];
    coltxt.recognizeCellTapped = NO;
    [cols addObject:coltxt];
    
    // Add Columns to Column collection
    dgrid.Columns = cols;
    
    // Assign UIDataGrid main properties
    dgrid.ColumnsWidth = COEFF_PAD * 110;
    // Datagrid's main colors
    dgrid.backgroundColor = [UIColor clearColor];
    dgrid.foregroundColor = [UIColor purpleColor];
    // Color of every other row
    dgrid.alternatingBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.1];
    // Specify properties of headers
    dgrid.HeaderBackgroundColor = [UIColor colorWithRed:0.1 green:0.2 blue:0.3 alpha:0.2];
    dgrid.HeaderForegroundColor = [UIColor whiteColor];
    dgrid.HeaderFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 8];
    dgrid.HeaderBorderColor = [UIColor lightGrayColor];
    dgrid.HeaderBorderWidth = COEFF_PAD * .3;
    dgrid.HeaderScrollHide = NO;
    
    dgrid.columnBorderColor = [UIColor cyanColor];
    dgrid.columnBorderWidth = COEFF_PAD * .5;
    dgrid.RowHeight = COEFF_PAD * 90;
    dgrid.datagridFont = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 4];
    dgrid.updateDataSource = YES;
    
    // Bind to populate data from DataSource into UIDataGrid object
    [dgrid DataBind];
}

// Click btnChange button
- (void)btnChangeClicked:(UIButton*)btn {
    // Change a particular cell of a particular column
    // Change second DataGridImageCell in very first column
    DataGridImageColumn *colImg = [dgrid.Columns objectAtIndex:0];
    DataGridImageCell *cellImg = [colImg.Cells objectAtIndex:1];
    BOOL changed = (cellImg.image.layer.borderColor == [UIColor redColor].CGColor);
    if (changed) {
        colImg.header.font = dgrid.HeaderFont;
        colImg.header.backgroundColor = colImg.HeaderBackgroundColor;
        colImg.header.textColor = [UIColor blueColor];
        cellImg.image.layer.borderColor = colImg.borderColor.CGColor;
        cellImg.image.layer.borderWidth = 0;
    } else {
        colImg.header.font = [UIFont fontWithName:@"Verdana" size:COEFF_PAD * 10];
        colImg.header.backgroundColor = [UIColor colorWithRed:0.6 green:0.1 blue:0.2 alpha:0.9];
        cellImg.image.layer.borderColor = [UIColor redColor].CGColor;
        cellImg.image.layer.borderWidth = COEFF_PAD * 3;
    }
    
    // Change second DataGridLabelCell in the second column
    DataGridLabelColumn *col = [dgrid.Columns objectAtIndex:1];
    DataGridLabelCell *cell = [col.Cells objectAtIndex:1];
    if (changed) {
        cell.label.backgroundColor = dgrid.backgroundColor;
        cell.label.textColor = col.textColor;
        cell.label.textAlignment = col.cellTemplate.textAlignment;
    } else {
        cell.label.backgroundColor = [UIColor lightGrayColor];
        cell.label.textColor = [UIColor blueColor];
        cell.label.textAlignment = NSTextAlignmentCenter;
    }
    
    // Depending on what Sample is loaded change corresponding cells
    if (btnPickerColumnSample.selected) {
        DataGridPickerColumn *colPkr = [dgrid.Columns objectAtIndex:2];
        DataGridPickerCell *cellPkr = [colPkr.Cells objectAtIndex:0];
        if (changed) {
            cellPkr.backgroundColor = dgrid.backgroundColor;
        } else {
            cellPkr.backgroundColor = [UIColor yellowColor];
        }
    } else if (btnDatePickerColumnSample.selected) {
        DataGridSliderColumn *colSld = [dgrid.Columns objectAtIndex:2];
        DataGridSliderCell *cellSld = [colSld.Cells objectAtIndex:0];
        if (changed) {
            cellSld.slider.tintColor = colSld.tintColor;
        } else {
            cellSld.slider.tintColor = [UIColor yellowColor];
        }
        
        DataGridDatePickerColumn *colPkr = [dgrid.Columns objectAtIndex:3];
        DataGridDatePickerCell *cellPkr = [colPkr.Cells objectAtIndex:0];
        if (changed) {
            cellPkr.backgroundColor = dgrid.backgroundColor;
        } else {
            cellPkr.backgroundColor = [UIColor yellowColor];
        }
    } else if (btnTextViewColumnSample.selected) {
        DataGridTextViewColumn *col = [dgrid.Columns objectAtIndex:2];
        DataGridTextViewCell *cell = [col.Cells objectAtIndex:0];
        if (changed) {
            cell.textView.backgroundColor = col.backgroundColor;
            cell.textView.textColor = col.textColor;
        } else {
            cell.textView.backgroundColor = [UIColor yellowColor];
            cell.textView.textColor = [UIColor redColor];
        }
    }
    return;
}

// Event-handling procedure that fires when a datagrid cell is tapped
- (void)datagridCellTapped:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex value:(NSString *)value {
    // Show a message
    [self msgbox:[NSString stringWithFormat:@"Value:%@", value] title:[NSString stringWithFormat:@"Column:%@ Row:%@", [NSString stringWithFormat:@"%d",colIndex], [NSString stringWithFormat:@"%d",rowIndex]]];
    return;
}

// Event-handling procedure that fires when a datagrid column's title is tapped
- (void)datagridColumnTitleTapped:(UIDataGrid *)datagrid colIndex:(int)colIndex title:(NSString *)title {
    // Show a message
    [self msgbox:[NSString stringWithFormat:@"Title:%@", title] title:[NSString stringWithFormat:@"Column:%@", [NSString stringWithFormat:@"%d",colIndex]]];
    return;
}

// Event-handling procedure that fires when a datagrid TextField Column cell's value is changed after editing
- (void)datagridTextFieldColumnCellEditingChanged:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UITextField *)sender {
    // Identify sender's DataGridTextFieldColumn
    DataGridTextFieldColumn *col = [dgrid.Columns objectAtIndex:colIndex];
    // Phone number formatting (works for 'Phone' field only)
    if ([col.Title isEqualToString:@"Phone"]) sender.text = [self phoneNumberFormat:sender.text];
    return;
}

// Event-handling procedure that fires when a datagrid TextField Column cell's starts editing
- (void)datagridTextFieldColumnCellEditingBegin:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UITextField *)sender {
    //==========
    // Put some commands here.
    //==========
    return;
}

// Event-handling procedure that fires when a datagrid TextField Column cell's ends editing
- (void)datagridTextFieldColumnCellEditingEnd:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UITextField *)sender {
    //==========
    // Put some commands here.
    //==========
    return;
}

// Event-handling procedure that fires when a datagrid TextView Column cell's value is changed
- (void)datagridTextViewColumnCellEditingChanged:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UITextView *)sender {
    // Identify sender's DataGridTextViewColumn
    DataGridTextViewColumn *col = [dgrid.Columns objectAtIndex:colIndex];
    // Identify index of datasource's data field that connected to the DataGridTextViewColumn
    int dataFieldIndex = col.dataFieldIndex;
    // Locate initial cell's data value in datagrid's data source
    NSMutableArray *itm = [[NSMutableArray alloc] init];
    itm = [dgrid.DataSource objectAtIndex:rowIndex];
    NSString *initValue = [itm objectAtIndex:dataFieldIndex];
    // Show a message
    [self msgbox:[NSString stringWithFormat:@"Initial:%@\rChanged:%@", initValue, sender.text] title:[NSString stringWithFormat:@"Column:%@", [NSString stringWithFormat:@"%d",colIndex]]];
    return;
}

// Event-handling procedure that fires when a datagrid Switch Column cell's value is changed
- (void)datagridSwitchColumnCellValueChanged:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UISwitch *)sender {
    // Show switch cell changed message
    [self msgbox:[NSString stringWithFormat:@"Row: %@\rChanged to: %@", [NSString stringWithFormat:@"%d",rowIndex], sender.on ? @"YES" : @"NO"] title:[NSString stringWithFormat:@"Switch Column: %@", [NSString stringWithFormat:@"%d",colIndex]]];
    return;
}

// Procedure that draws every item in each PickerColumn cell populating from pickerDataSource
- (UIView *)datagridPickerColumnCellViewForRow:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex rowInPicker:(NSInteger)row component:(NSInteger)component reusingView:(UIView *)view sender:(UIPickerView *)pickerView {
    // Define a custom UILabel object
    UILabel *lblRow = [[UILabel alloc] init];
    // Identify the picker column object
    DataGridPickerColumn *col = [dgrid.Columns objectAtIndex:colIndex];
    lblRow.frame = pickerView.frame;
    [lblRow setTextAlignment:NSTextAlignmentCenter];
    // Define the picker cell content
    NSString *rowItem = [col.pickerDataSource objectAtIndex:row];
    [lblRow setTextColor:[UIColor purpleColor]];
    // Add the text
    [lblRow setText:rowItem];
    return lblRow;
}

// Procedure that runs every time when an item in a PickerColumn cell has been selected
- (void)datagridPickerColumnCellRowSelected:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex rowInPicker:(NSInteger)row component:(NSInteger)component sender:(UIPickerView *)pickerView {
    
    DataGridPickerColumn *col = [dgrid.Columns objectAtIndex:colIndex];
    NSMutableArray *itm = [dgrid.DataSource objectAtIndex:rowIndex];
    NSString *str = [itm objectAtIndex:col.dataFieldIndex];
    DataGridImageColumn *colImg = [dgrid.Columns objectAtIndex:0];
    DataGridImageCell *cellImg = [colImg.Cells objectAtIndex:rowIndex];
    [cellImg.image setImage:[UIImage imageNamed:[NSString stringWithFormat:@"%@%@", [str lowercaseString], @".jpg"]]];
    DataGridLabelColumn *colLbl = [dgrid.Columns objectAtIndex:1];
    DataGridLabelCell *cellLbl = [colLbl.Cells objectAtIndex:rowIndex];
    cellLbl.label.text = str;
    return;
}

// Event-handling procedure that fires when a datagrid Slider Column cell's value is changed
- (void)datagridSliderColumnCellValueChanged:(UIDataGrid *)datagrid rowIndex:(int)rowIndex colIndex:(int)colIndex sender:(UISlider *)sender {
    // Identify the Label column object
    DataGridLabelColumn *colLbl = [dgrid.Columns objectAtIndex:1];
    // Identify the Label cell object
    DataGridLabelCell *cellLbl = [colLbl.Cells objectAtIndex:rowIndex];
    // Change the Label's text with Slider's value
    cellLbl.label.text = [@(sender.value) description];
    return;
}

// Event-handling procedure that fires when datagrid scrolling started
- (void)datagridScrollViewDidScroll:(UIDataGrid *)datagrid scrollView:(UIScrollView *)scrollView {
    //==========
    // Put some commands here.
    //==========
    return;
}

// Generate UIDataGrid object's DataSources
// 1. NSMutableArray DataSource
- (NSMutableArray *)generateNSMutableArrayDataSource {
    NSMutableArray *dsource = [[NSMutableArray alloc] init];
    NSMutableArray *itm = [[NSMutableArray alloc] init];
    [itm addObject:@"1"];
    [itm addObject:@"Mitsubishi"];
    [itm addObject:@"1"];
    [itm addObject:@"3"];
    [itm addObject:@"2/20/2004"];
    [itm addObject:@"3/6/2004"];
    [itm addObject:@"000-000-0000"];
    [itm addObject:@"true"];
    [itm addObject:@"mitsubishi.jpg"];
    [itm addObject:@"The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas]; two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins]; a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night."];
    [itm addObject:@"100010001"];
    [itm addObject:@"100"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"2"];
    [itm addObject:@"Dodge"];
    [itm addObject:@"2"];
    [itm addObject:@"17"];
    [itm addObject:@"2/20/2004"];
    [itm addObject:@"3/6/2004"];
    [itm addObject:@"111-111-1111"];
    [itm addObject:@"false"];
    [itm addObject:@"dodge.jpg"];
    [itm addObject:@"Anti-Lock Brakes]; Air Bags]; AM/FM Stereo Radio]; Automatic Transmission]; Air Conditioning]; 4 or 6 Cylinder Engine]; Power-Assist Steering]; Tinted Windows]; Dual Mirrors]; Tilt Steering Wheel]; Power Top]; Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment)]; a bar with lighting]; a wash-stand and shelves. For creation of comfort in the compartments]; they are isolated from the salon by one common and two personal doors. Moreover passengers space]; in the carriage there are 2 washing-rooms with cold and hot water]; kitchen]; a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio]; video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night."];
    [itm addObject:@"000200002"];
    [itm addObject:@"10"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"3"];
    [itm addObject:@"Toyota"];
    [itm addObject:@"3"];
    [itm addObject:@"40"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"222-222-2222"];
    [itm addObject:@"true"];
    [itm addObject:@"toyota.jpg"];
    [itm addObject:@"The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas]; two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins]; a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night."];
    [itm addObject:@"123451234"];
    [itm addObject:@"50"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"4"];
    [itm addObject:@"Honda"];
    [itm addObject:@"1"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"honda.jpg"];
    [itm addObject:@"Anti-Lock Brakes]; Air Bags]; AM/FM Stereo Radio]; Automatic Transmission]; Air Conditioning]; 4 or 6 Cylinder Engine]; Power-Assist Steering]; Tinted Windows]; Dual Mirrors]; Tilt Steering Wheel]; Power Top]; Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment)]; a bar with lighting]; a wash-stand and shelves. For creation of comfort in the compartments]; they are isolated from the salon by one common and two personal doors. Moreover passengers space]; in the carriage there are 2 washing-rooms with cold and hot water]; kitchen]; a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio]; video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night."];
    [itm addObject:@"123451234"];
    [itm addObject:@"99"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"5"];
    [itm addObject:@"Jeep"];
    [itm addObject:@"2"];
    [itm addObject:@"59"];
    [itm addObject:@""];
    [itm addObject:@"3/6/2004"];
    [itm addObject:@""];
    [itm addObject:@"true"];
    [itm addObject:@"jeep.jpg"];
    [itm addObject:@"The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas]; two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins]; a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night."];
    [itm addObject:@""];
    [itm addObject:@"90"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"6"];
    [itm addObject:@"Subaru"];
    [itm addObject:@"1"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"subaru.jpg"];
    [itm addObject:@"Anti-Lock Brakes]; Air Bags]; AM/FM Stereo Radio]; Automatic Transmission]; Air Conditioning]; 4 or 6 Cylinder Engine]; Power-Assist Steering]; Tinted Windows]; Dual Mirrors]; Tilt Steering Wheel]; Power Top]; Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment)]; a bar with lighting]; a wash-stand and shelves. For creation of comfort in the compartments]; they are isolated from the salon by one common and two personal doors. Moreover passengers space]; in the carriage there are 2 washing-rooms with cold and hot water]; kitchen]; a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio]; video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night."];
    [itm addObject:@""];
    [itm addObject:@"80"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"7"];
    [itm addObject:@"Volkswagen"];
    [itm addObject:@"1"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"true"];
    [itm addObject:@"volkswagen.JPG"];
    [itm addObject:@"The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas]; two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins]; a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night."];
    [itm addObject:@""];
    [itm addObject:@"70"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"8"];
    [itm addObject:@"Ford"];
    [itm addObject:@"2"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"ford.jpg"];
    [itm addObject:@"Anti-Lock Brakes]; Air Bags]; AM/FM Stereo Radio]; Automatic Transmission]; Air Conditioning]; 4 or 6 Cylinder Engine]; Power-Assist Steering]; Tinted Windows]; Dual Mirrors]; Tilt Steering Wheel]; Power Top]; Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment)]; a bar with lighting]; a wash-stand and shelves. For creation of comfort in the compartments]; they are isolated from the salon by one common and two personal doors. Moreover passengers space]; in the carriage there are 2 washing-rooms with cold and hot water]; kitchen]; a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio]; video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night."];
    [itm addObject:@"999999999"];
    [itm addObject:@"60"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"9"];
    [itm addObject:@"Nissan"];
    [itm addObject:@"3"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"true"];
    [itm addObject:@"nissan.jpg"];
    [itm addObject:@"The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas]; two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins]; a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night."];
    [itm addObject:@""];
    [itm addObject:@"50"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"10"];
    [itm addObject:@"Chevrolet"];
    [itm addObject:@"1"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"chevrolet.jpg"];
    [itm addObject:@"Anti-Lock Brakes]; Air Bags]; AM/FM Stereo Radio]; Automatic Transmission]; Air Conditioning]; 4 or 6 Cylinder Engine]; Power-Assist Steering]; Tinted Windows]; Dual Mirrors]; Tilt Steering Wheel]; Power Top]; Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the"];
    [itm addObject:@""];
    [itm addObject:@"40"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"11"];
    [itm addObject:@"Fiat"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"fiat.jpg"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"30"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"12"];
    [itm addObject:@"Hyundai"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"hyundai.jpg"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"35"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"13"];
    [itm addObject:@"Kia"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"kia.jpg"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"29"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"14"];
    [itm addObject:@"Mercedes"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"mercedes.jpg"];
    [itm addObject:@"Nice!"];
    [itm addObject:@""];
    [itm addObject:@"20"];
    [itm addObject:@"0"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"15"];
    [itm addObject:@"Peugeot"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"peugeot.jpg"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"100"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    itm = [[NSMutableArray alloc] init];
    [itm addObject:@"16"];
    [itm addObject:@"Renault"];
    [itm addObject:@"0"];
    [itm addObject:@"0"];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@""];
    [itm addObject:@"false"];
    [itm addObject:@"renault.jpg"];
    [itm addObject:@"from France"];
    [itm addObject:@""];
    [itm addObject:@"100"];
    [itm addObject:@"1"];
    [dsource addObject:itm];
    return dsource;
}

// 3. JSON file DataSource
- (NSMutableArray *)generateJSONfileDataSource {
    NSMutableArray *dsource = [[NSMutableArray alloc] init];
    NSString *filePath =[[NSBundle mainBundle] pathForResource:@"Cars" ofType:@"json"];
    NSString *fileContents =[NSString stringWithContentsOfFile:filePath encoding:NSUTF8StringEncoding error:NULL];
    NSArray *SourceArr = (NSArray *)[NSJSONSerialization JSONObjectWithData:[fileContents dataUsingEncoding:NSUTF16StringEncoding] options:0 error:NULL];
    for (NSDictionary *row in SourceArr) {
        NSMutableArray *itm = [[NSMutableArray alloc] init];
        [itm addObject:row[@"ID"]];
        [itm addObject:row[@"Corporation"]];
        [itm addObject:row[@"CarID"]];
        [itm addObject:row[@"ModelID"]];
        [itm addObject:row[@"DateFirst"]];
        [itm addObject:row[@"TimeFirst"]];
        [itm addObject:row[@"Phone"]];
        [itm addObject:row[@"Mark"]];
        [itm addObject:row[@"Picture"]];
        [itm addObject:row[@"Remark"]];
        [itm addObject:row[@"ZipCode"]];
        [itm addObject:row[@"Availability"]];
        [itm addObject:row[@"Accessibility"]];
        [dsource addObject:itm];
    }
    return dsource;
}

// 4. XML file DataSource
- (NSMutableArray *)generateXMLfileDataSource {
    NSMutableArray *dsource = [[NSMutableArray alloc] init];
    NSString *filePath =[[NSBundle mainBundle] pathForResource:@"Cars" ofType:@"xml"];
    NSString *fileContents =[NSString stringWithContentsOfFile:filePath encoding:NSUTF8StringEncoding error:NULL];
    NSDictionary *dict = [[[XMLReader XMLtoDictionary:fileContents error:NULL] objectForKey:@"table"] objectForKey:@"Cars"];
    for (NSDictionary *row in dict) {
        NSMutableArray *itm = [[NSMutableArray alloc] init];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"ID"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Corporation"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"CarID"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"ModelID"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"DateFirst"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"TimeFirst"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Phone"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Mark"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Picture"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Remark"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"ZipCode"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Availability"][@"text"]]];
        [itm addObject:[NSString stringWithFormat:@"%@", [row objectForKey:@"Accessibility"][@"text"]]];
        [dsource addObject:itm];
    }
    return dsource;
}

// Format Phone Number in TextField cell
-(NSString*) phoneNumberFormat:(NSString*)initNumber {
    if (initNumber.length == 0) return @"";
    // Use regex to remove non-digits (including spaces) - just the numbers
    NSError *error = NULL;
    NSRegularExpression *regex = [NSRegularExpression regularExpressionWithPattern:@"[\\s-\\(\\)]" options:NSRegularExpressionCaseInsensitive error:&error];
    initNumber = [regex stringByReplacingMatchesInString:initNumber options:0 range:NSMakeRange(0, [initNumber length]) withTemplate:@""];
    
    // Check if the number is to long - remove last extra chars
    if (initNumber.length > 10) initNumber = [initNumber substringToIndex:10];
    
    // Format the number: 123 456 7890. If it is less then 7 digits then use this regex
    if (initNumber.length < 7) {
        initNumber = [initNumber stringByReplacingOccurrencesOfString:@"(\\d{3})(\\d+)"
                                                           withString:@"($1) $2"
                                                              options:NSRegularExpressionSearch
                                                                range:NSMakeRange(0, [initNumber length])];
        
    } else {
        initNumber = [initNumber stringByReplacingOccurrencesOfString:@"(\\d{3})(\\d{3})(\\d+)"
                                                           withString:@"($1) $2-$3"
                                                              options:NSRegularExpressionSearch
                                                                range:NSMakeRange(0, [initNumber length])];
    }
    return initNumber;
}

// Show message box
- (void)msgbox:(NSString *)msg title:(NSString *)title {
    UIAlertView *alert = [[UIAlertView alloc] initWithTitle:title message:msg delegate:self cancelButtonTitle:@"Ok" otherButtonTitles:nil, nil];
	[alert show];
}

@end





















